document.addEventListener('DOMContentLoaded', () => {
    const userForm = document.getElementById('user-form');
    const loggedInUserSpan = document.getElementById('logged-in-user');
    const logoutBtn = document.getElementById('logout-btn');
    const loginForm = document.getElementById('login-form');
    const taskControls = document.getElementById('task-controls');
    const addTaskForm = document.getElementById('add-task-form');
    const addCategoryForm = document.getElementById('add-category-form');
    const editCategorySelect = document.getElementById('edit-category-select');
    const newCategoryNameInput = document.getElementById('new-category-name');
    const editCategoryBtn = document.getElementById('edit-category-btn');
    const deleteCategoryBtn = document.getElementById('delete-category-btn');
    const taskTableBody = document.querySelector('#task-table tbody');
    const searchTaskInput = document.getElementById('search-task');
    const filterByDateInput = document.getElementById('filter-by-date');
    const exportTasksButton = document.getElementById('export-tasks');
    const tableHeaders = document.querySelectorAll('#task-table th[data-sort]');

    let tasks = JSON.parse(localStorage.getItem('tasks')) || [];
    let categories = JSON.parse(localStorage.getItem('categories')) || [];
    let loggedInUser = localStorage.getItem('loggedInUser') || '';
    let currentSort = { key: '', ascending: true };

    const loadCategories = () => {
        const categorySelect = document.getElementById('task-category');
        categorySelect.innerHTML = '';
        editCategorySelect.innerHTML = '';
        categories.forEach(category => {
            const option = document.createElement('option');
            option.value = category;
            option.textContent = category;
            categorySelect.appendChild(option);
            const editOption = document.createElement('option');
            editOption.value = category;
            editOption.textContent = category;
            editCategorySelect.appendChild(editOption);
        });
    };

    const saveTasks = () => {
        localStorage.setItem('tasks', JSON.stringify(tasks));
        renderTasks();
    };

    const saveCategories = () => {
        localStorage.setItem('categories', JSON.stringify(categories));
        loadCategories();
    };

    const renderTasks = (tasksToRender = tasks) => {
        taskTableBody.innerHTML = '';
        tasksToRender.forEach((task, index) => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${task.name}</td>
                <td>${task.category}</td>
                <td>${task.startDate}</td>
                <td>${task.dueDate}</td>
                <td>${task.completedDate || ''}</td>
                <td><button onclick="editTask(${index})"><img src="image/edit.png" alt="rediget" height="20px"></button></td>
                <td><button onclick="deleteTask(${index})"><img src="image/dzest.png" alt="dzest" height="20px"></button></td>
                <td><button onclick="completeTask(${index})"><img src="image/izdarits.png" alt="izdarits" height="20px"></button></td>
            `;
            taskTableBody.appendChild(row);
        });
    };

    userForm.addEventListener('submit', (e) => {
        e.preventDefault();
        loggedInUser = document.getElementById('username').value;
        localStorage.setItem('loggedInUser', loggedInUser);
        displayLoggedInUser();
    });

    logoutBtn.addEventListener('click', () => {
        loggedInUser = '';
        localStorage.removeItem('loggedInUser');
        displayLoggedInUser();
    });

    const displayLoggedInUser = () => {
        if (loggedInUser) {
            loginForm.style.display = 'none';
            taskControls.style.display = 'block';
            loggedInUserSpan.textContent = `Sveiks, ${loggedInUser}`;
            logoutBtn.style.display = 'inline';
        } else {
            loginForm.style.display = 'block';
            taskControls.style.display = 'none';
            loggedInUserSpan.textContent = '';
            logoutBtn.style.display = 'none';
        }
    };

    addTaskForm.addEventListener('submit', (e) => {
        e.preventDefault();
        const taskName = document.getElementById('task-name').value;
        const taskStartDate = document.getElementById('task-start-date').value;
        const taskDueDate = document.getElementById('task-due-date').value;
        const taskCategory = document.getElementById('task-category').value;

        tasks.push({
            name: taskName,
            startDate: taskStartDate,
            dueDate: taskDueDate,
            category: taskCategory,
            completedDate: null
        });

        saveTasks();
        addTaskForm.reset();
    });

    addCategoryForm.addEventListener('submit', (e) => {
        e.preventDefault();
        const categoryName = document.getElementById('category-name').value;
        if (!categories.includes(categoryName)) {
            categories.push(categoryName);
            saveCategories();
        }
        addCategoryForm.reset();
    });

    editCategoryBtn.addEventListener('click', () => {
        const oldCategoryName = editCategorySelect.value;
        const newCategoryName = newCategoryNameInput.value;
        if (newCategoryName && oldCategoryName) {
            const categoryIndex = categories.indexOf(oldCategoryName);
            categories[categoryIndex] = newCategoryName;
            tasks.forEach(task => {
                if (task.category === oldCategoryName) {
                    task.category = newCategoryName;
                }
            });
            saveCategories();
            saveTasks();
        }
    });

    deleteCategoryBtn.addEventListener('click', () => {
        const categoryName = editCategorySelect.value;
        categories = categories.filter(category => category !== categoryName);
        tasks = tasks.filter(task => task.category !== categoryName);
        saveCategories();
        saveTasks();
    });

    searchTaskInput.addEventListener('input', (e) => {
        const query = e.target.value.toLowerCase();
        const filteredTasks = tasks.filter(task => task.name.toLowerCase().includes(query));
        renderTasks(filteredTasks);
    });

    filterByDateInput.addEventListener('input', (e) => {
        const queryDate = e.target.value;
        const filteredTasks = tasks.filter(task => task.startDate === queryDate);
        renderTasks(filteredTasks);
    });

    exportTasksButton.addEventListener('click', () => {
        const filteredTasks = getFilteredTasks();
        const csvHeader = "Name,Category,Start Date,Due Date,Completed Date\n";
        const csvRows = filteredTasks.map(task => [
            task.name,
            task.category,
            task.startDate,
            task.dueDate,
            task.completedDate || ''
        ].join(',')).join('\n');
        
        const csvData = csvHeader + csvRows;
        const blob = new Blob([csvData], { type: 'text/csv' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'darbi.csv';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
    });

    tableHeaders.forEach(header => {
        header.addEventListener('click', () => {
            const key = header.getAttribute('data-sort');
            sortTasks(key);
        });
    });

    const sortTasks = (key) => {
        const isAscending = currentSort.key === key ? !currentSort.ascending : true;
        tasks.sort((a, b) => {
            const aValue = key === 'completedDate' ? (a[key] || Infinity) : a[key];
            const bValue = key === 'completedDate' ? (b[key] || Infinity) : b[key];
            return isAscending ? aValue.localeCompare(bValue) : bValue.localeCompare(aValue);
        });
        currentSort = { key, ascending: isAscending };
        saveTasks();
    };

    const renderFilteredTasks = (tasksToRender) => {
        taskTableBody.innerHTML = '';
        tasksToRender.forEach((task, index) => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${task.name}</td>
                <td>${task.category}</td>
                <td>${task.startDate}</td>
                <td>${task.dueDate}</td>
                <td>${task.completedDate || ''}</td>
                <td><button onclick="editTask(${index})">Rediģēt</button></td>
                <td><button onclick="deleteTask(${index})">Izdzēst</button></td>
                <td><button onclick="completeTask(${index})">Atzīmēt izpildi</button></td>
            `;
            taskTableBody.appendChild(row);
        });
    };

    const getFilteredTasks = () => {
        const query = searchTaskInput.value.toLowerCase();
        const queryDate = filterByDateInput.value;
        return tasks.filter(task => task.name.toLowerCase().includes(query) && (queryDate ? task.startDate === queryDate : true));
    };

    window.editTask = (index) => {
        const task = tasks[index];
        document.getElementById('task-name').value = task.name;
        document.getElementById('task-start-date').value = task.startDate;
        document.getElementById('task-due-date').value = task.dueDate;
        document.getElementById('task-category').value = task.category;
        deleteTask(index);
    };

    window.deleteTask = (index) => {
        tasks.splice(index, 1);
        saveTasks();
    };

    window.completeTask = (index) => {
        tasks[index].completedDate = new Date().toISOString().split('T')[0];
        saveTasks();
    };

    loadCategories();
    displayLoggedInUser();
    renderTasks();
});